% This script does the following:
%
% -- Runs a random walk mixed layer model to predict Be-10 concentrations
% at the end of a long period of continuous exposure and soil mixing
%
% -- Then takes the same particles, imposes a wildfire history, and
% predicts Ne-21 production and loss by heating. 
%
% It does this for a series of scenarios in which wildfire heating turns on
% at different times in the exposure history. This illustrates the effect
% of a non-steady-state fire history in modifying the depth profile of
% diffusively mobile nuclides. 
%
% Accompanies a paper called 'Short communication: Cosmogenic noble gas
% depletion in soils by wildfire heating,' by Greg Balco, Alan Hidy, Will
% Struble, and Josh Roering.
%
% Written by Greg Balco, June 2023. 
%
% Note: this script loads a data file called 'data21.mat' that contains
% measured Ne-21 concentrations. These are only needed for making a nice
% looking plot in lines 170-176. If you don't have this file or don't care,
% just comment out those lines. 

% Clean up 

clear all; close all;

%% Part 1: mixing model to predict Be-10 concentrations

% Parameters for mixing model

% Fixed parameters

d.mlh = 170; % Mixed layer thickness (cm)
d.dt = 250; % timestep (yr)
d.rho = 1.7; % Density. This is average density from pit/auger measurements for top 170 cm. 
d.ni = 5000; % Number of particles to track
d.l10 = 4.99e-7; % Be-10 decay constant (/yr)
d.P10 = 4.01.*1.27; % Be-10 production rate. From Stone/Lal scaling for Cougar Pass site.

% Parameters that can be adjusted to fit the Be-10 data

bump = 10; % Standard deviation of random walk step (cm). Note: this 
% isn't independent of the time step. The actual controlling parameter is
% bump./d.dt, so you can change either bump or d.dt to get the same effect.
t = 425000; % Exposure duration (yr)

% Also specify Be-10 concentrations for plotting

d10 = [879446	21067	0	10
858258	16820	20	30
873237	28211	30	40
837782	15639	52	61
950255	41191	61	70
874016	15951	70	79
854637	15940	79	89
836332	28953	89	99
861879	16150	99	109
824445	15475	129	139
841551	18051	159	169
611631	11489	203	213
491995	9472	234	244
339091	8930	274	284];

% Generate time domain
nt = round(t./d.dt); % number of timesteps; this should be an integer anyway
% Generate a time vector in years BP (for decay correction)
tv = fliplr(0:d.dt:t);

% Generate random starting depths for particles
% Note: these are in cm
init_z = rand(1,d.ni) .* d.mlh; 

% Generate random series of steps for all particles
bumps = randn(d.ni,nt).*bump;
% Initialize z matrix
z = zeros(d.ni,nt);
z = [init_z' z]; % insert initial points at beginning of matrix
% Rows are particles, columns are timesteps

% Also initialize decay correction matrix
decay = zeros(size(z));

% Step through each particle trajectory and implement bounce conditions.
% Particles "bounce" off upper and lower boundaries of mixed layer. Like,
% if you are at depth 5 and the step in this jump is -6 cm, then you go up
% 5 cm, bounce, and go back down 1 cm. 
% There is probably a faster way to vectorize this. 

for a = 1:d.ni % loop through rows (particles)
    for b = 1:nt % loop through columns (timesteps)
        z(a,b+1) = z(a,b) + bumps(a,b); % Apply step b
        % bounce conditions
        if z(a,b+1) < 0; z(a,b+1) = abs(z(a,b+1)); end
        if z(a,b+1) > d.mlh; z(a,b+1) = d.mlh - (z(a,b+1)-d.mlh); end    
        if z(a,b+1) < 0; z(a,b+1) = abs(z(a,b+1)); end % why is this here twice? 
        if z(a,b+1) < 0 | z(a,b+1) > d.mlh; error('Out of bounds -- ??'); end
    end
    % Also populate decay matrix by rows while we are looping.
    decay(a,:) = exp(-d.l10.*tv);
end

% Now compute Be-10 concentrations produced in each time step and corrected
% for decay to present time
Lsp = 150; % effective attenuation length for spallogenic production. Note: ignore muons
% Note: ignore decay during each time step, e.g., linearize

N10_in_step = d.P10*exp(-z.*d.rho./Lsp).*d.dt.*decay;

% Sum each row to arrive at Be-10 concentration at end of run. 

endN10 = sum(N10_in_step'); % Note: this winds up as a row vector. 

% Now average Be-10 concentrations at 10 cm. 

% Define bin boundaries
num_bins = floor(d.mlh./10);
bin_bottoms = 10:10:(10.*num_bins);
bin_tops = 0:10:(10.*(num_bins - 1));
bin_mids = (bin_tops + bin_bottoms)./2;

% Obtain depth of each particle at end of run
endz = z(:,end);

% Initialize
averages = zeros(size(bin_mids));

% Average particles in each bin
for a = 1:length(bin_mids)
	to_average = find(endz >= bin_tops(a) & endz < bin_bottoms(a));
	averages(a) = mean(endN10(to_average));        
end 

% Also pad the data out to 300 with simple exponential approximation. 
pad_bins = bin_bottoms(end)+10:10:300;
pad_Ns = (d.P10.*exp(-pad_bins.*d.rho/Lsp)./d.l10).*(1-exp(-d.l10.*t));

% Package reaults for later use

out.z = z;
out.endz = endz;
out.tv = tv;
out.bin_mids = bin_mids;
out.bin_mids_padded = [bin_mids pad_bins];
out.bin_tops = bin_tops;
out.bin_bottoms = bin_bottoms;
out.N10pred = averages;
out.N10pred_padded = [averages pad_Ns];

% Plot Be-10 data and model
for a = 1:size(d10,1)
    xx = [-1 1].*d10(a,2) + d10(a,1);
    yy = [1 1].*((d10(a,3)+d10(a,4))./2);
    plot(xx,yy,'k'); hold on;
end
plot(d10(:,1),(d10(:,3)+d10(:,4))./2,'ko','markerfacecolor',[0.5 0.5 0.5],'markersize',9);
set(gca,'ydir','reverse');
xlabel('Nuclide concentration (atoms/g)');
ylabel('Depth (cm)');

plot(out.N10pred_padded,out.bin_mids_padded,'k'); grid on;
drawnow;


%% Part 2: Ne-21 production and degassing

% Load Ne-21 data from file and add to plot. 

load data21;
for a = 1:length(d21.mds)
    xx = [-1 1].*d21.SEs(a) + d21.EWMs(a); 
    yy = [1 1].*d21.mds(a);
    plot(xx.*1e6,yy,'k');
end
plot(d21.EWMs.*1e6,d21.mds,'ko','markerfacecolor','r');

plot(get(gca,'xlim'),[170 170],'k--');
drawnow;

% Now start production/diffusion calculations. 

% First obtain diffusion kinetics
% Assume grain is like the one heated in Shuster and Farley (2005)
% These are diffusion kinetics for Ne-21 from that paper
Ea = 154; % Activation energy kJ/mol
% Ea = 84.5; % corresponding value for He-3
Ea_J_mol = Ea.*1000; % Convert to J/mol
lnD0a2 = 16; % ln(1/s)
% lnD0a2 = 11; % corresponding value for He-3
D0a2 = exp(lnD0a2); % 1/s
R = 8.314; % J/degK/mol
a = 0.0215; % grain radius (cm). This is just the grain analyzed by S & F. 

% Next choose fire heating parameters
fire_surface_temp_C = 450; % Surface temperature during a fire
fire_heating_L = 6; % Exponential attenuation length for temperature (cm)
fire_duration = 60.*60; % One hour in seconds

% Use those to generate fire temperature matrices matching the z matrix
fire_temp_C = fire_surface_temp_C.*exp(-out.z./fire_heating_L); % some temp at surface, exponential with 6 cm L
fire_temp_K = fire_temp_C + 273.15;

% Also specify inherited Ne-21 concentration. This is unconstrained for
% these samples but could be measured in a deeper borehole.
init_N21 = 2e6; % Atoms/g quartz

% Now compute diffusivity D/a2 during each step using Arrhenius equation
Da2 = D0a2.*exp(-Ea_J_mol./(R.*fire_temp_K));
% Compute amount of Dt/a2 experienced in each step, assuming one fire per
% time step. 
Dta2_step = Da2.*fire_duration; 

% Now we are going to compute production and degassing for various
% transient scenarios. Basically we turn on the fire regime described above
% at different times in the model. 

% Define model time steps at which to turn on fires in each simulation. 
% These are chosen to give a nice spread of results. 

% Note: these assume the 425 ka scenario. If you change that you also have 
% to change these to get the transient scenarios in the right place. 
start_step = [1 901 1101 1301 1501 1601 1701];
start_time = out.tv(start_step); % figure corresponding time before present

% Loop through all transient scenarios. 
for b = 1:length(start_step)
    disp(int2str(start_step(b)));
    % Impose zeros in first part of heating matrix such that there is no
    % diffusive loss until specified start time in this scenario. 
    
    Dta2_step(:,1:start_step(b)) = zeros(d.ni,start_step(b));

    % Now cumulate Dt/a2. 
    
    Dta2_cum = fliplr(cumsum(fliplr(Dta2_step),2)); 
    % This is the cumulative Dt/a2 that the Ne-21 produced in each step has
    % experienced between that step and the end of the run. 

    % Now calculate loss of that stuff. This is the series approximation
    % from Fechtig and Kalbitzer. This is somewhat time-consuming. You
    % could use the various approximations instead. 
    
    loss = zeros(size(out.z)); % initialize
    
    o = 1024; % Number of series terms -- More than 1000 is probably best
    for a = 1:d.ni
        this = Dta2_cum(a,:);
        % Equation 4 of Fechtig and Kalbitzer
        % Compute Bt
        Bt = this.*(pi^2);
        [ttt,nnn] = meshgrid(Bt,1:o);
        % Sum for each time step
        part1 = sum((nnn.^(-2)).*exp(-(nnn.^2).*ttt));
        loss(a,:) = 1 - (6./(pi.^2)).*part1; % This gives cumulative fraction RELEASED
        if a./1000 == round(a./1000); disp(int2str(a)); end
    end

    % Now that we have the loss fractions (which are equivalent to the
    % 'decay' matrix for Be-10), calculate how much Ne-21 there is. 
    
    N21_in_step = d.P10*exp(-out.z.*d.rho./Lsp).*d.dt.*4.03; % Atoms Ne-21 produced in each step
    
    % Also, "produce" all inherited Ne-21 in first step. The point is that that
    % inventory can diffuse out for the entire model run...so the end
    % concentration can be lower than the inherited concentration. 
    N21_in_step(:,1) = zeros(d.ni,1) + init_N21; 

    % Account for loss of Ne-21 produced in each step during the 
    % remainder of the model run. 
    N21_in_step_heated = N21_in_step.*(1-loss); % Loss was release fxn, so 1-that
    cumN21_heated = sum(N21_in_step_heated'); % Sum up

    % Compute average concentration in each depth bin
    for a = 1:length(out.bin_mids)
        to_average = find(out.endz > out.bin_tops(a) & out.endz < out.bin_bottoms(a));
        averages_heated(a) = mean(cumN21_heated(to_average));        
    end  

    % Plot results of this iteration
    plot(averages_heated,out.bin_mids,'r');
    this_text = int2str(round(start_time(b)./1000))
    if b == 5; this_text = [this_text ' ka'];end
    text(averages_heated(4),out.bin_mids(4),this_text,'backgroundcolor','w','horizontalalignment','center','verticalalignment','middle');
    drawnow;
    N21pred{b} = averages_heated;
end

% Save data for later plotting

save current_results out N21pred start_time






